/*
 * MeleeAttackInfo.h
 *
 * Created 9/6/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/6/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Only CombatCharacter type Objects can melee attack. Make sure to supply a 
  * CombatCharacter type when creating an ActionTask or EffectTask using
  * a MeleeAttackInfo.
  */
 
 #ifndef MELEE_ATTACK_INFO_H
 #define MELEE_ATTACK_INFO_H
 
 template <typename T> class MeleeAttackInfo;
 
 #include "global.h"
 
 #include "AttackInfo.h"
 #include "PushInfo.h"
 #include "DamageObject.h"
 #include "pointerTo.h"
 
 /**
  * Class specification for MeleeAttackInfo
  */
 template <typename T>
 class MeleeAttackInfo : public AttackInfo<T>, public PushInfo<T>
 {
 // Static Functions
 private:
    static inline AsyncTask::DoneStatus process_melee_attack( GenericAsyncTask* task_Ptr, void* data_Ptr );
    
 // Data Members
 private:
    
 // Local Functions
 public:
    MeleeAttackInfo( DamageObject<T>* dmg_obj_Ptr = NULL, 
                const double duration_before_attaching_dmg_obj = 0.0, 
                const double duration_before_detaching_dmg_obj = 0.0,
                const double stun_duration = 0.0,
                const MovementInfoCollection<T>& movement_info_collection = MovementInfoCollection<T>() );
    MeleeAttackInfo( const MeleeAttackInfo<T>& melee_attack_info );
    virtual ~MeleeAttackInfo();
    inline MeleeAttackInfo<T>& operator=( const MeleeAttackInfo<T>& melee_attack_info );
    virtual inline void split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** STATIC FUNCTIONS **/
 
 /**
  * process_melee_attack() handles attaching the DamageObject to a CharacterObject.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask.
  * The ActionInfo of the ActionTask must also be a MeleeAttackInfo.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus MeleeAttackInfo<T>::process_melee_attack( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    nassertr( task_Ptr != NULL, AsyncTask::DS_done );
    
    PT(ActionTask<T>) action_task_Ptr( reinterpret_cast<ActionTask<T>*>( task_Ptr ) );
    PT(MeleeAttackInfo<T>) melee_attack_info_Ptr( dynamic_cast<MeleeAttackInfo<T>*>( action_task_Ptr->get_action_info() ) );
    nassertr( melee_attack_info_Ptr != NULL, AsyncTask::DS_done );
    PT(Object<T>) object_Ptr( action_task_Ptr->get_object() );
    nassertr( object_Ptr != NULL, AsyncTask::DS_done );
    
    double elapsed_time( global::_clock_Ptr->get_real_time() - melee_attack_info_Ptr->get_time_action_was_invoked() );
    
    // if the elapsed time has reached for attaching the DamageObject
    if ( elapsed_time > melee_attack_info_Ptr->get_duration_before_attaching_damage_object() )
    {
        melee_attack_info_Ptr->get_damage_object()->reparent_to( *object_Ptr, melee_attack_info_Ptr->get_time_action_was_invoked() + melee_attack_info_Ptr->get_duration_before_attaching_damage_object() );
    }
    
    // if the elapsed time has reached for detaching the DamageObject
    if ( elapsed_time > melee_attack_info_Ptr->get_duration_before_detaching_damage_object() )
    {
        //melee_attack_info_Ptr->get_damage_object()->detach_node();
        return AsyncTask::DS_done;
    }
    
    return AsyncTask::DS_cont;
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 MeleeAttackInfo<T>::MeleeAttackInfo( DamageObject<T>* dmg_obj_Ptr, 
                                      const double duration_before_attaching_dmg_obj, 
                                      const double duration_before_detaching_dmg_obj,
                                      const double stun_duration,
                                      const MovementInfoCollection<T>& movement_info_collection )
               : AttackInfo<T>( dmg_obj_Ptr, duration_before_attaching_dmg_obj, duration_before_detaching_dmg_obj ),
                 PushInfo<T>( stun_duration, movement_info_collection )
 {
    nassertv( dmg_obj_Ptr != NULL );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 MeleeAttackInfo<T>::MeleeAttackInfo( const MeleeAttackInfo<T>& melee_attack_info )
                    : AttackInfo<T>( melee_attack_info ),
                      PushInfo<T>( melee_attack_info )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 MeleeAttackInfo<T>::~MeleeAttackInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified MeleeAttackInfo to this MeleeAttackInfo.
  *
  * @param (const MeleeAttackInfo<T>&) melee_attack_info
  * @return MeleeAttackInfo<T>&
  */
 template <typename T>
 inline MeleeAttackInfo<T>& MeleeAttackInfo<T>::operator=( const MeleeAttackInfo<T>& melee_attack_info )
 {
    AttackInfo<T>::operator=( melee_attack_info );
    PushInfo<T>::operator=( melee_attack_info );
    
    return *this;
 }
 
 /**
  * split_tasks() either assigns the specified task to the specified 
  * task manager or splits up the task into multiple tasks before 
  * assigning the tasks to the specified task manager.
  *
  * @param (ActionTask<T>*) action_task_Ptr
  * @param (AsyncTaskManager*) task_mgr_Ptr
  */
 template <typename T>
 inline void MeleeAttackInfo<T>::split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr )
 {
    nassertv( action_task_Ptr != NULL );
    nassertv( task_mgr_Ptr != NULL );
    
    PushInfo<T>::split_tasks( action_task_Ptr, task_mgr_Ptr );
    
    if ( AttackInfo<T>::get_duration_before_detaching_damage_object() > 0.0 )
    {
        // Create the ActionTask to be added to the AsyncTaskManager
        PT(ActionTask<T>) new_action_task_Ptr( new ActionTask<T>( action_task_Ptr->get_object(), &MeleeAttackInfo<T>::process_melee_attack, action_task_Ptr->get_action_info() ) );
        new_action_task_Ptr->set_upon_death( &AttackInfo<T>::upon_death_handler );
        
        //new_action_task_Ptr->add_to_task_manager( task_mgr_Ptr );
        task_mgr_Ptr->add( new_action_task_Ptr );
    }
 }
 
 #endif // MELEE_ATTACK_INFO_H